<?php
/**
 * User Profile Field
 *
 * @package     AutomatorWP\Integrations\BuddyBoss\Filters\User_Profile_Field
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyBoss_User_Profile_Field_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'buddyboss';
    public $filter = 'buddyboss_user_profile_field';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User profile field', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>profile field</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Meta key. %2$s: Condition. %1$s: Meta value. */
            'edit_label'        => sprintf( __( '%1$s %2$s %3$s', 'automatorwp-pro' ), '{field_name}', '{condition}', '{field_value}'  ),
            /* translators: %1$s: Meta key. %2$s: Condition. %1$s: Meta value. */
            'log_label'         => sprintf( __( '%1$s %2$s %3$s', 'automatorwp-pro' ), '{field_name}', '{condition}', '{field_value}' ),
            'options'           => array(
                'field_name' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'field_name',
                    'name'              => __( 'Field:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'field name', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Field ID.', 'automatorwp-pro' )
                        . ' ' . __( 'The field ID can be found in the URL when editing a profile field, in the part of the URL that says "&field_id={FIELD_ID}", where {FIELD_ID} should be a number that corresponds to the field ID.', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_buddyboss_get_profile_fields',
                    'options_cb'        => 'automatorwp_buddyboss_options_cb_profile_field',
                    'default'           => 'any'
                ) ),
                'condition' => automatorwp_utilities_condition_option(),
                'field_value' => array(
                    'from' => 'field_value',
                    'default' => __( 'value', 'automatorwp-pro' ),
                    'fields' => array(
                        'field_value' => array(
                            'name' => __( 'Value:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                    ),
                ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Shorthand
        $field_name = $filter_options['field_name'];
        $condition = $filter_options['condition'];
        $field_value = $filter_options['field_value'];

        // Bail if wrong configured
        if( empty( $field_name ) ) {
            $this->result = __( 'Filter not passed. Field option has not been configured.', 'automatorwp-pro' );
            return false;
        }

        if( ! function_exists( 'xprofile_get_field_data' ) ) {
            $this->result = __( 'Filter not passed. Could not access to the function "xprofile_get_field_data".', 'automatorwp-pro' );
            return false;
        }

        $user_field_value = xprofile_get_field_data( $field_name, $user_id, 'comma' );

        // Don't deserve if meta value doesn't match with the user meta value
        if( ! automatorwp_condition_matches( $user_field_value, $field_value, $condition ) ) {
            $this->result = sprintf( __( 'Filter not passed. User %1$s has the value "%2$s" and does not meets the condition %3$s "%4$s".', 'automatorwp-pro' ),
                automatorwp_buddyboss_get_profile_field_title( $field_name ),
                $user_field_value,
                automatorwp_utilities_get_condition_label( $condition ),
                $field_value
            );
            return false;
        }

        $this->result = sprintf( __( 'Filter passed. User %1$s has the value "%2$s" and meets the condition %3$s "%4$s".', 'automatorwp-pro' ),
            automatorwp_buddyboss_get_profile_field_title( $field_name ),
            $user_field_value,
            automatorwp_utilities_get_condition_label( $condition ),
            $field_value
        );

        return $deserves_filter;

    }

}

new AutomatorWP_BuddyBoss_User_Profile_Field_Filter();